/*
 * SCRFMain.cpp - v1.0 Noose
 * Functions to store and retrieve standard code from a resource
 * file.
 * The resource file is a standard text file. It has a series of
 * Tags to identify the following text. These tags are ALWAYS
 * preceded by "//". The tags currently supported are:
 *
 * // Owner:        What function owns the following data
 * // Function:     The label of the following function
 * // Creator:      Who created this then?
 * // Description   What does it do?
 * // Code          Identifies the start of any code (terminated by
 *                  any of the other //'s
 * // Data          The start of any data for the block
 * // Unicode       As with code but for unicode source
 * // Ansicode      As with code but for ANSI source
 * // End           Marks the end of the resource file
 *
 * In this version vcl functions are include (namely MessageBox), I'll
 * be removing these later on - so be warned. This will mean there will
 * be no messages telling you that you've tried to open a file which
 * doesn't exist or cannot be opened for some other reason
 */

#include "scrf.hpp"
#include <stdio.h>
#include <string.h>


// Truncates a string (removes all white space & eoln chars)
void TruncateString(char *aString) {
    int Index;

    Index=strlen(aString)-1;
    while ( (Index>=0) && (aString[Index]<=' ') )
        aString[Index--]='\0';
} // TruncateString(...)

/*
 * TCodeLine Class definitions
 * 
 * TCodeLine is used to define a single line of code.
 * It provides a linked-list facility so that the TCodeElement
 * functions need only refer to the start of the list
 *
 */

// Constructor for the TCodeLine class
// Creates a new line and fills it with the specified text
TCodeLine::TCodeLine(char *SomeText) {
    _Text=new char[strlen(SomeText)+1];
    strcpy(_Text,SomeText);
    TruncateString(_Text);
    _Prev=NULL; _Next=NULL;
} // TCodeLine::TCodeLine(...)

TCodeLine::~TCodeLine() {
//    printf("  Deleting line.\n");
    delete(_Text);
} // TCodeLine::~TCodeLine()

// Functions to return the private data
char *TCodeLine::Text() { return _Text; }
TCodeLine *TCodeLine::Prev() { return _Prev; }
TCodeLine *TCodeLine::Next() { return _Next; }

// Functions to set the private data
void TCodeLine::Prev(TCodeLine *p) { _Prev=p; }
void TCodeLine::Next(TCodeLine *n) { _Next=n; }


// Function to return a pointer to the last line in a list
TCodeLine *LastLine(TCodeLine *_LineList) {
    TCodeLine *CurrLine;

    CurrLine=_LineList;
    if (CurrLine!=NULL) {
        while (CurrLine->Next()!=NULL) 
            CurrLine=CurrLine->Next();
    }
    return CurrLine;
} // LastLine(...)

/*
 * TCodeElement Class definitions
 * A TCodeElement is a single block of code (a collection of
 * CodeLines) with a label to identify it. From the point of
 * creation, the code can then be referenced by the label.
 *
 */

// Constructor for the TCodeElement class
// Creates a single code element with no prev/next ptrs or lines
TCodeElement::TCodeElement(char *cn,char *o) {
    _CodeName=new char[strlen(cn)+1];
    _Owner=new char[strlen(o)+1];
    strcpy(_CodeName,cn);
    strcpy(_Owner,o);

    _Prev=NULL; _Next=NULL;

    _SourceCode=NULL;
    _Description=NULL;
    _Creator=NULL;
    _DC=NULL;
    _LinesOfData=NULL;

    _numLines=0;
    _descLines=0;
    _dataLines=0;

    _CodeType=TYPE_NONE;
} // TCodeElement::TCodeElement(...)

// Destructor for the TCodeElement class
TCodeElement::~TCodeElement() {
   if (_CodeName)
       delete(_CodeName);
   if (_Creator)
       delete(_Creator);
   if (_Owner)
       delete(_Owner);
   if (_DC)
       delete(_DC);
   DeleteLines(_SourceCode);
   DeleteLines(_LinesOfData);
   DeleteLines(_Description);
} // TCodeElement::~TCodeElement()

void TCodeElement::DeleteLines(TCodeLine *CurrLine) {
    TCodeLine *PrevLine;

    while (CurrLine!=NULL) {
        PrevLine=CurrLine;
        CurrLine=CurrLine->Next();
        delete(PrevLine);
    }
} // TCodeElement::DeleteLines(..)

// Functions to return the private data
char *TCodeElement::CodeName(void) { return _CodeName; }
unsigned long TCodeElement::numLines(void) { return _numLines; }
unsigned long TCodeElement::descLines(void) { return _descLines; }
unsigned long TCodeElement::dataLines(void) { return _dataLines; }
char *TCodeElement::Owner(void) { return _Owner; }

int TCodeElement::Type(void) { return _CodeType; }
void TCodeElement::Type(int ct) { _CodeType=ct; }

TCodeElement *TCodeElement::Prev(void) { return _Prev; }
TCodeElement *TCodeElement::Next(void) { return _Next; }

char *TCodeElement::Creator() { return _Creator; }
char *TCodeElement::DCreator() { return _DC; }

// Functions to set the private data
void TCodeElement::Prev(TCodeElement *p) { _Prev=p; }
void TCodeElement::Next(TCodeElement *n) { _Next=n; }

void TCodeElement::Creator(char *c) {
    _Creator=new char[strlen(c)+1];
    strcpy(_Creator,c);
} // TCodeElement::Creator(...)

void TCodeElement::DCreator(char *c) {
    if (c!=NULL) {
        _DC=new char[strlen(c)+1];
        strcpy(_DC,c);
    } else
        _DC=NULL;
} // TCodeElement::Creator(...)

TCodeLine *TCodeElement::Desc(unsigned long aLine) {
    TCodeLine *CurrLine,*PrevLine;
    unsigned long LineNum;

    CurrLine=_Description;
    PrevLine=NULL;
    LineNum=FIRST_LINE;

    while ( (CurrLine!=NULL) && (LineNum<aLine) ) {
        PrevLine=CurrLine;
        CurrLine=CurrLine->Next();
        LineNum++;
    }

    if (CurrLine==NULL) CurrLine=PrevLine;

    return CurrLine;
} // TCodeElement::Desc(...)

TCodeLine *TCodeElement::Data(unsigned long aLine) {
    TCodeLine *CurrLine,*PrevLine;
    unsigned long LineNum;

    CurrLine=_LinesOfData;
    PrevLine=NULL;
    LineNum=FIRST_LINE;

    while ( (CurrLine!=NULL) && (LineNum<aLine) ) {
        PrevLine=CurrLine;
        CurrLine=CurrLine->Next();
        LineNum++;
    }

    if (CurrLine==NULL) CurrLine=PrevLine;

    return CurrLine;
} // TCodeElement::Data(...)

// Function to return a pointer to a specified line
TCodeLine *TCodeElement::Line(unsigned long aLine) {
    TCodeLine *CurrLine,*PrevLine;
    unsigned long LineNum;

    CurrLine=_SourceCode;
    PrevLine=NULL;
    LineNum=FIRST_LINE;

    while ( (CurrLine!=NULL) && (LineNum<aLine) ) {
        PrevLine=CurrLine;
        CurrLine=CurrLine->Next();
        LineNum++;
    }

    if (CurrLine==NULL) CurrLine=PrevLine;

    return CurrLine;
} // TCodeElement::Line(...)

void TCodeElement::AddDesc(char *TextOnLine) {
    TCodeLine *TempLine,*CurrLine;

    TempLine=new TCodeLine(TextOnLine);

    CurrLine=LastLine(_Description);
    if (CurrLine==NULL) {
        _Description=TempLine;
        TempLine->Next(NULL);
        TempLine->Prev(NULL);
    } else {
        CurrLine->Next(TempLine);
        TempLine->Prev(CurrLine);
        TempLine->Next(NULL);
    }
    _descLines++;
} // TCodeElement::AddDesc(...)

void TCodeElement::AddData(char *TextOnLine) {
    TCodeLine *TempLine,*CurrLine;

    TempLine=new TCodeLine(TextOnLine);

    CurrLine=LastLine(_LinesOfData);
    if (CurrLine==NULL) {
        _LinesOfData=TempLine;
        TempLine->Next(NULL);
        TempLine->Prev(NULL);
    } else {
        CurrLine->Next(TempLine);
        TempLine->Prev(CurrLine);
        TempLine->Next(NULL);
    }
    _dataLines++;
} // TCodeElement::AddData(...)

// Function to insert a new line at a specific point
void TCodeElement::InsertLine(unsigned long aLine,char *TextOnLine,int Where) { 
    TCodeLine *CurrLine,*TempLine,*PrevLine,*NextLine;

    TempLine=new TCodeLine(TextOnLine);

    switch(Where) {
        case INSERT_AT_END:
            CurrLine=LastLine(_SourceCode);
            if (CurrLine==NULL) {
                _SourceCode=TempLine;
                TempLine->Next(NULL);
                TempLine->Prev(NULL);
            } else {
                CurrLine->Next(TempLine);
                TempLine->Prev(CurrLine);
                TempLine->Next(NULL);
            }
            break;
        case INSERT_BEFORE:
            CurrLine=Line(aLine);
            if (CurrLine==NULL) {
                _SourceCode=TempLine;
                TempLine->Next(NULL);
                TempLine->Prev(NULL);
            } else {
                PrevLine=CurrLine->Prev();
                if (PrevLine==NULL) {
                    _SourceCode=TempLine;
                    TempLine->Prev(NULL);
                } else 
                    PrevLine->Next(TempLine);
                CurrLine->Prev(TempLine);
                TempLine->Prev(PrevLine);
                TempLine->Next(CurrLine);
            }
            break;
        case INSERT_AFTER:
            CurrLine=Line(aLine);
            if (CurrLine==NULL) {
                _SourceCode=TempLine;
                TempLine->Next(NULL);
                TempLine->Prev(NULL);
            } else {
                NextLine=CurrLine->Next();
                if (NextLine==NULL) {
                    CurrLine->Next(TempLine);
                    TempLine->Prev(CurrLine);
                } else {
                    CurrLine->Next(TempLine);
                    NextLine->Prev(TempLine);
                    TempLine->Prev(CurrLine);
                }
                TempLine->Next(NextLine);
            }
            break;
    }
    _numLines++;
} // TCodeElement::InsertLine(...)


// Function to check the existence of a file
int FileExists(char *fn) {
    FILE *Handle;

    Handle=fopen(fn,"rb");
    if (Handle!=NULL) {
        fclose(Handle);
        return TRUE;
    } else
        return FALSE;
} // FileExists(...)

// Returns a pointer to the last element in a list
TCodeElement *LastElement(TCodeElement *_CodeList) {
    TCodeElement *CurrCode;

    CurrCode=_CodeList;
    if (CurrCode!=NULL) {
        while (CurrCode->Next()!=NULL)
            CurrCode=CurrCode->Next();
    }
    return CurrCode;
} // LastElement(...)

// Returns the type of line specified by LineString
int LineType(char *LineString) {
    int Index,i;
    char TabName[256];

    for (Index=0;(LineString[Index]==' ') || (LineString[Index]=='\t');Index++);   // Skip leading spaces

    if ( (LineString[Index]=='/') && (LineString[Index+1]=='/') ) {
        for (Index=2;(LineString[Index]==' ') || (LineString[Index]=='\t');Index++);

        i=0;
        for (;(LineString[Index]!=':') && (LineString[Index]>=' ');Index++)
            TabName[i++]=LineString[Index];
        TabName[i]='\0';
        strupr(TabName);
        TruncateString(TabName);

        if (!strcmp(TabName,"FUNCTION"))
            return FUNCTION_NAME;
        if (!strcmp(TabName,"CODE"))
            return CODE_START;
        if (!strcmp(TabName,"UNICODE"))
            return UNICODE_START;
        if (!strcmp(TabName,"ANSICODE"))
            return ANSICODE_START;
        if (!strcmp(TabName,"DATA"))
            return DATA_START;
        if (!strcmp(TabName,"DESCRIPTION"))
            return DESCRIPTION_TEXT;
        if (!strcmp(TabName,"CREATOR"))
            return CREATOR_NAME;
        if (!strcmp(TabName,"OWNER"))
            return OWNER_NAME;
        if (!strcmp(TabName,"END"))
            return END_FLAG;
    }
    return OTHER_LINE;
} // LineType(...)

/*
 * TSourceCode Class definitions
 * The TSourceCode class holds all the current blocks of code
 * as a linked-list. When created, all of the source code which
 * is defined as being owned by the caller is loaded. More source
 * code can be loaded if desired by defining the Owner and names
 * of function which are required.
 *
 */

// Constructor for the TSourceCode Class
TSourceCode::TSourceCode(char *fn,char *o) {

    _SourceCodeList=NULL;

    strcpy(_fileName,fn);
    strcpy(_Owner,o);

    _numFunctions=0;

    if (FileExists(_fileName)) {
        LZInput=new LZAri(_fileName,NULL);
        BuildData("");
        delete(LZInput);
        LZInput=NULL;
    }
} // TSourceCode::TSourceCode(...)

// Destructor for the TSourceCode Class
TSourceCode::~TSourceCode() {
    TCodeElement *CurrElem,*PrevElem;

    CurrElem=_SourceCodeList;
    while (CurrElem!=NULL) {
        PrevElem=CurrElem;
        CurrElem=CurrElem->Next();
        delete(PrevElem);
    }
} // TSourceCode::~TSourceCode()

// Returns the field contents (what follows the ':')
void TSourceCode::FieldContents(char *line,char *Buffer) {
    int Index,i;

    for (Index=0;line[Index]!=':';Index++);
    for (Index++;line[Index]<=' ';Index++);

    for (i=0;line[Index+i]>' ';i++)
        Buffer[i]=line[Index+i];
    Buffer[i]='\0';
    TruncateString(Buffer);
} // TSourceCode::FieldContents(...)

// The main routine - reads and builds the source code list
void TSourceCode::BuildData(char *fname) {
    char CurrLine[256],OwnerCopy[MAX_OWNER_LEN],OwnerFromFile[MAX_OWNER_LEN];
    char CurrentFName[MAX_FUNCNAME_LEN],CurrentCreator[MAX_CREATOR_LEN];
    char DefaultCreator[MAX_CREATOR_LEN];
    TCodeElement *CurrElem,*PrevElem;
    int ReadThisOne=TRUE,TopLineType;
    unsigned long i;

    LZInput->Readln(CurrLine);

    if (strlen(fname)!=0)
        ReadThisOne=FALSE;

    strcpy(OwnerCopy,_Owner);
    strupr(OwnerCopy);

    CurrElem=NULL;

    while ( !(LZInput->eof()) ) {
        switch(LineType(CurrLine)) {
            case OWNER_NAME:
                strupr(CurrLine);
                FieldContents(CurrLine,OwnerFromFile);
                CurrentFName[0]='\0';
                DefaultCreator[0]='\0';
                if (strlen(_Owner)==0)
                    strcpy(OwnerCopy,OwnerFromFile);
                if (!strcmp(OwnerFromFile,OwnerCopy) ) {
                    LZInput->Readln(CurrLine);
                    while ( (LineType(CurrLine)!=OWNER_NAME) && (LineType(CurrLine)!=END_FLAG)
                            && !(LZInput->eof()) ) {
                        TopLineType=LineType(CurrLine);
                        switch (TopLineType) {
                            case FUNCTION_NAME:
                                CurrentCreator[0]='\0';
                                FieldContents(CurrLine,CurrentFName);

                                if ( (strlen(fname)==0) || (!strcmp(CurrentFName,fname)) ) {
                                    ReadThisOne=TRUE;
                                    _numFunctions++;
                                    CurrElem=new TCodeElement(CurrentFName,OwnerCopy);
                                } else
                                    ReadThisOne=FALSE;

                                if (DefaultCreator[0]!='\0')
                                    CurrElem->DCreator(DefaultCreator);
                                else
                                    CurrElem->DCreator(NULL);

                                LZInput->Readln(CurrLine);
                                break;
                            case DESCRIPTION_TEXT:
                                if (ReadThisOne) {
                                    if (CurrElem!=NULL) {
                                        LZInput->Readln(CurrLine);
                                        while (LineType(CurrLine)==OTHER_LINE) {
                                            CurrElem->AddDesc(CurrLine);
                                            LZInput->Readln(CurrLine);
                                        }
                                    }
                                } else
                                    LZInput->Readln(CurrLine);
                                break;
                            case CREATOR_NAME:
                                FieldContents(CurrLine,CurrentCreator);
                                if (CurrentFName[0]=='\0') 
                                    strcpy(DefaultCreator,CurrentCreator);
                                else if (ReadThisOne)
                                    CurrElem->Creator(CurrentCreator);

                                LZInput->Readln(CurrLine);
                                break;
                            case CODE_START:
                            case UNICODE_START:
                            case ANSICODE_START:
                                if (ReadThisOne) {
                                    if ( (CurrentCreator[0]=='\0') && (DefaultCreator[0]!='\0') )
                                        CurrElem->Creator(DefaultCreator);
    
                                    LZInput->Readln(CurrLine);
        
                                    while (LineType(CurrLine)==OTHER_LINE) {
                                        CurrElem->InsertLine(1,CurrLine,INSERT_AT_END);
                                        LZInput->Readln(CurrLine);
                                    }
                                    if (TopLineType==UNICODE_START)
                                        CurrElem->Type(TYPE_UNICODE);
                                    else if (TopLineType==ANSICODE_START) 
                                        CurrElem->Type(TYPE_ANSI);
                                    else
                                        CurrElem->Type(TYPE_NONE);
                                    Add(CurrElem);
                                    TopLineType=LineType(CurrLine);
                                    if ( (TopLineType==UNICODE_START) || (TopLineType==ANSICODE_START) || (TopLineType==CODE_START) ) {
                                        PrevElem=CurrElem;
                                        CurrElem=new TCodeElement(CurrentFName,OwnerCopy);
                                        CurrElem->Creator(CurrentCreator);
                                        for (i=1;i<=PrevElem->descLines();i++)
                                            CurrElem->AddDesc(PrevElem->Desc(i)->Text());
                                        for (i=1;i<=PrevElem->dataLines();i++)
                                            CurrElem->AddData(PrevElem->Data(i)->Text());
                                        ++_numFunctions;
                                    }
                                } else
                                    LZInput->Readln(CurrLine);
                                break;
                            case DATA_START:
                                if (ReadThisOne) {
                                    LZInput->Readln(CurrLine);

                                    while (LineType(CurrLine)==OTHER_LINE) {
                                        CurrElem->AddData(CurrLine);
                                        LZInput->Readln(CurrLine);
                                    }
                                } else
                                    LZInput->Readln(CurrLine);
                                break;
                            default:
                                LZInput->Readln(CurrLine);
                                break;
                        } //switch
                    } //while
                } //if
                break;
            default:
                LZInput->Readln(CurrLine);
                break;
        } //switch
    } //while
} // TSourceCode::BuildData()

// Functions to return the private data
char *TSourceCode::fileName(void) { return _fileName; }

// Returns the a ptr to the TCodeElement with the label specified

TCodeElement *TSourceCode::Block(char *in,int CodeType) {
    char FNameCopy[MAX_FUNCNAME_LEN];
    int Flag;
    TCodeElement *CurrElem;

    strcpy(FNameCopy,in);
    strupr(FNameCopy);
    TruncateString(FNameCopy);
    Flag=FALSE;

    CurrElem=_SourceCodeList;
    do {
        if (Flag==TRUE)
            CurrElem=CurrElem->Next();
        Flag=TRUE;
        while ( (CurrElem!=NULL) && (strcmp(FNameCopy,strupr(strdup(CurrElem->CodeName())))) )
            CurrElem=CurrElem->Next();
    } while ( (CurrElem!=NULL) && (CurrElem->Type()!=CodeType) );
    return CurrElem;
} // TSourceCode::Block(...)

TCodeElement *TSourceCode::Block(char *in) {
    return Block(in,TYPE_NONE);
} // TSourceCode::Block(...)

// Adds a new TCodeElement to the list
void TSourceCode::Add(TCodeElement *aCodeElem) {
    TCodeElement *PrevCode;

    PrevCode=LastElement(_SourceCodeList);
    if (PrevCode==NULL) {
        _SourceCodeList=aCodeElem;
        aCodeElem->Prev(NULL);
    } else {
        aCodeElem->Prev(PrevCode);
        PrevCode->Next(aCodeElem);
    }
    aCodeElem->Next(NULL);
} // TSourceCode::Add(...)

void TSourceCode::AddFromOther(char *o,char *f) {
    strcpy(_Owner,o);

    LZInput=new LZAri(_fileName,NULL);
    BuildData(f);
    delete(LZInput);
    LZInput=NULL;
} // TSourceCode::AddFromOther(...)

void TSourceCode::AddFromOther(char *o) {
  	AddFromOther(o,"");
} // TSourceCode::AddFromOther(...)

unsigned long TSourceCode::numFunctions() { return _numFunctions; }

char *TSourceCode::FunctionList(int CodeType,char *o) {
    unsigned long i,TotalLength,Index;
    TCodeElement *CurrentElement;
    char *BufferPtr,*OName;

    TotalLength=0;
    OName=new char[strlen(o)+1];
    strcpy(OName,o);
    strupr(OName);
    CurrentElement=_SourceCodeList;
    Index=0;

    if (CurrentElement!=NULL) {
        for (i=0; i<_numFunctions; i++) {
            if (CurrentElement->Type()==CodeType)
                TotalLength+=(strlen(CurrentElement->CodeName())+1);
            CurrentElement=CurrentElement->Next();
        }
        TotalLength++;

        BufferPtr=new char[TotalLength];
        CurrentElement=_SourceCodeList;

        for (i=0; i<_numFunctions; i++) {
	    	if (strlen(o)==0) {
    	    	delete (OName);
        	    OName=new char [strlen(CurrentElement->Owner())+1];
        		strcpy(OName,CurrentElement->Owner());
	        }
            if ( (CurrentElement->Type()==CodeType) && (!strcmp(CurrentElement->Owner(),OName)) ) {
                strcpy(BufferPtr+Index,CurrentElement->CodeName());
                Index+=(strlen(BufferPtr+Index)+1);
            }
            CurrentElement=CurrentElement->Next();
        }
    } else
        BufferPtr=new char[1];
    BufferPtr[Index]='\0';
    delete(OName);
    return BufferPtr;
} // TSourceCode::FunctionList(...)

char *TSourceCode::FunctionList(char *o) {
	return FunctionList(TYPE_NONE,o);
}

char *TSourceCode::FunctionList() {
    return FunctionList(TYPE_NONE,"");
} // TSourceCode::FunctionList()

int TSourceCode::FoundInBuffer(char *Start,char *Text,unsigned long Len) {
    unsigned long Index;
    int Found;

    Index=0;
    Found=FALSE;
    while ( (Index<Len) && (Found!=TRUE) ) {
        if (!strcmp(Start+Index,Text))
            Found=TRUE;
        Index+=strlen(Start+Index)+1;
    }
    return Found;
} // TSourceCode::FoundInBuffer(..)

char *TSourceCode::OwnerList() {
    char *Buffer,*StartOfBuffer;
    TCodeElement *anElem;
    unsigned long CurrStrLen,CurrLen,i;

    anElem=_SourceCodeList;
    Buffer=NULL;
    StartOfBuffer=NULL;
    CurrLen=0;

    while (anElem!=NULL) {
        CurrStrLen=strlen(anElem->Owner());
        if (Buffer==NULL) {
            Buffer=new char[CurrStrLen+1];
            strcpy(Buffer,anElem->Owner());
            StartOfBuffer=Buffer;
            CurrLen=CurrStrLen+1;
        } else {
            if (!FoundInBuffer(StartOfBuffer,anElem->Owner(),CurrLen)) {
                Buffer=new char[CurrLen+CurrStrLen+1];
                for (i=0;i<CurrLen;i++)
                    Buffer[i]=StartOfBuffer[i];
                strcpy(Buffer+CurrLen,anElem->Owner());
                delete (StartOfBuffer);
                StartOfBuffer=Buffer;
                CurrLen=CurrLen+CurrStrLen+1;
            }
        }
        anElem=anElem->Next();
    }

    Buffer=new char[CurrLen+1];
    if (StartOfBuffer!=NULL) {
        for (i=0;i<CurrLen;i++)
            Buffer[i]=StartOfBuffer[i];
        delete (StartOfBuffer);
    }
    StartOfBuffer=Buffer;
    StartOfBuffer[CurrLen]='\0';

    return (StartOfBuffer);
} // TSourceCode::OwnerList()

unsigned long TSourceCode::CreateBuffer(char *Buffer) {
    char *Owners,*FunctionList,*DC;
    char LineBuff[256];
    unsigned long Index,TotalLength,FunctionListLen,i,a;
    int DefaultAcc,NoWriteFlag;
    TCodeElement *CurrElem;

    if (Buffer==NULL) {
        NoWriteFlag=TRUE;
        Buffer=LineBuff;
    } else
        NoWriteFlag=FALSE;

    Owners=OwnerList();

    Index=0;
    TotalLength=0;
    while (Owners[Index]!='\0') {       // For each Owner
        sprintf(Buffer,"// Owner: %s\r\n",Owners+Index);
        TotalLength+=strlen(Buffer);    // Length for Owner
        if (!NoWriteFlag)
            Buffer+=strlen(Buffer);

        // Calculate the length of a buffer for holding function names
        CurrElem=_SourceCodeList;
        FunctionListLen=0;
        for (i=0;i<_numFunctions;i++) {
            FunctionListLen+=strlen(CurrElem->CodeName())+1;
            CurrElem=CurrElem->Next();
        }
        // Create the buffer
        FunctionList=new char[FunctionListLen+1];
        // Create the list of all individual (i.e. not repeated) function names
        FunctionListLen=0;
        CurrElem=_SourceCodeList;
        i=0;
        DefaultAcc=FALSE;
        while (CurrElem!=NULL) {        // For each Function
            if (!strcmp(CurrElem->Owner(),Owners+Index)) { // With this owner
                if (!FoundInBuffer(FunctionList,CurrElem->CodeName(),FunctionListLen)) {
                    DC=CurrElem->DCreator();
                    if ( (DC!=NULL) && (!DefaultAcc) ) {
                        sprintf(Buffer,"// Creator: %s\r\n",DC);
                        TotalLength+=strlen(Buffer);
                        if (!NoWriteFlag) 
                            Buffer+=strlen(Buffer);
                        DefaultAcc=TRUE;
                    }

                    strcpy(FunctionList+i,CurrElem->CodeName());
                    i+=strlen(CurrElem->CodeName())+1;
                    FunctionListLen=i;      // Variable Overkill methinks
                    sprintf(Buffer,"// Function: %s\r\n",CurrElem->CodeName());
                    TotalLength+=strlen(Buffer);
                    if (!NoWriteFlag) 
                        Buffer+=strlen(Buffer);
                    // Find the length of the creator (if exists)
                    if (CurrElem->Creator()!=NULL) {
                        if ( ( (DefaultAcc) && (strcmp(DC,CurrElem->Creator()) ) 
                            || (DC==NULL) ) ) {
                            sprintf(Buffer,"// Creator: %s\r\n",CurrElem->Creator());
                            TotalLength+=strlen(Buffer);
                            if (!NoWriteFlag)
                                Buffer+=strlen(Buffer);
                        }
                    }
                    // Find the length of the description
                    if (CurrElem->descLines()>0) {
                        sprintf(Buffer,"// Description\r\n");
                        TotalLength+=strlen(Buffer);
                        if (!NoWriteFlag)
                            Buffer+=strlen(Buffer);
                    }
                    for (a=1;a<=CurrElem->descLines();a++) {
                        sprintf(Buffer,"%s\r\n",CurrElem->Desc(a)->Text());
                        TotalLength+=strlen(Buffer);
                        if (!NoWriteFlag) 
                            Buffer+=strlen(Buffer);
                    }
                    // Find the length of the associated data
                    if (CurrElem->dataLines()>0) {
                        sprintf(Buffer,"// Data\r\n");
                        TotalLength+=strlen(Buffer);
                        if (!NoWriteFlag)
                            Buffer+=strlen(Buffer);
                    }
                    for (a=1;a<=CurrElem->dataLines();a++) {
                        sprintf(Buffer,"%s\r\n",CurrElem->Data(a)->Text());
                        TotalLength+=strlen(Buffer);
                        if (!NoWriteFlag)
                            Buffer+=strlen(Buffer);
                    }
                }
                // Find the length of the code tag
                if (CurrElem->Type()==TYPE_NONE) {
                    sprintf(Buffer,"// Code\r\n");
                    TotalLength+=strlen(Buffer);
                    if (!NoWriteFlag)
                        Buffer+=strlen(Buffer);
                } else if (CurrElem->Type()==TYPE_UNICODE) {
                    sprintf(Buffer,"// Unicode\r\n");
                    TotalLength+=strlen(Buffer);
                    if (!NoWriteFlag)
                        Buffer+=strlen(Buffer);
                } else if (CurrElem->Type()==TYPE_ANSI) {
                    sprintf(Buffer,"// Ansicode\r\n");
                    TotalLength+=strlen(Buffer);
                    if (!NoWriteFlag)
                        Buffer+=strlen(Buffer);
                }
                // Find the length of the associated code
                for (a=1;a<=CurrElem->numLines();a++) {
                    sprintf(Buffer,"%s\r\n",CurrElem->Line(a)->Text());
                    TotalLength+=strlen(Buffer);
                    if (!NoWriteFlag)
                        Buffer+=strlen(Buffer);
                }
            }
            CurrElem=CurrElem->Next();
        }
        Index+=strlen(Owners+Index)+1;
        delete (FunctionList);
    }
    // And now add the end tag (and the trailing 0)
    sprintf(Buffer,"// End\r\n");
    TotalLength+=strlen(Buffer)+1;
    delete (Owners);
    return (TotalLength);
}

void TSourceCode::OutputSCR(char *fn) {
    char *Buffer;
    LZAri *anLZAri;
    unsigned long BufferLen;

    BufferLen=CreateBuffer(NULL);
    Buffer=new char[BufferLen+1];
    CreateBuffer(Buffer);
    anLZAri=new LZAri(fn,Buffer);
    delete(anLZAri);
    delete(Buffer);
} // TSourceCode::OutputSCR(..)
